<?php

namespace TC\PaygateCrocoPay\Payment;

use Exception;
use GuzzleHttp\Exception\GuzzleException;
use XF;
use XF\Entity\PaymentProfile;
use XF\Entity\PurchaseRequest;
use XF\Http\Request;
use XF\Mvc\Controller;
use XF\Mvc\Reply\AbstractReply;
use XF\Payment\AbstractProvider;
use XF\Payment\CallbackState;
use XF\Purchasable\Purchase;

class CrocoPay extends AbstractProvider
{
	protected string $requestKey;

	/**
	 * @return string
	 */
	public function getTitle(): string
	{
		return 'CrocoPay';
	}

	/**
	 * @return string
	 */
	public function getApiEndpoint(): string
	{
		return 'https://crocopay.tech/api/v2/';
	}

	protected function getVariableName(): string
	{
		return XF::$versionId >= 2010070 ? 'form_params' : 'body';
	}

	/**
	 * @param array $options
	 * @param array $errors
	 *
	 * @return bool
	 */
	public function verifyConfig(array &$options, &$errors = []): bool
	{
		if (empty($options['client_id']) || empty($options['client_secret']))
		{
			$errors[] = XF::phrase('tc_pg_crocopay_you_must_provide_all_data');
		}

		return !$errors;
	}

	/**
	 * @param PurchaseRequest $purchaseRequest
	 * @param Purchase        $purchase
	 *
	 * @return array
	 */
	protected function getPaymentParams(PurchaseRequest $purchaseRequest, Purchase $purchase): array
	{
		$this->requestKey = $purchaseRequest->request_key;
		$profileOptions = $purchase->paymentProfile->options;

		return [
			'client_id'     => $profileOptions['client_id'],
			'client_secret' => $profileOptions['client_secret'],
			'amount'        => number_format(
				$purchaseRequest->cost_amount,
				2,
				'.',
				''
			),
			'currency'      => $purchaseRequest->cost_currency,
			'successUrl'    => $purchase->returnUrl,
			'cancelUrl'     => $purchase->cancelUrl,
			'callbackUrl'   => $this->getCallbackUrl()
		];
	}

	public function getCallbackUrl(): string
	{
		if (!empty($this->requestKey))
		{
			return sprintf('%s/payment_callback.php?_xfProvider=%s&request_key=%s',
				XF::app()->options()->boardUrl,
				$this->providerId,
				$this->requestKey
			);
		}

		return parent::getCallbackUrl();
	}

	/**
	 * @param Controller      $controller
	 * @param PurchaseRequest $purchaseRequest
	 * @param Purchase        $purchase
	 *
	 * @return AbstractReply
	 * @throws GuzzleException
	 */
	public function initiatePayment(Controller $controller, PurchaseRequest $purchaseRequest, Purchase $purchase): XF\Mvc\Reply\AbstractReply
	{
		$payment = $this->getPaymentParams($purchaseRequest, $purchase);
		try
		{
			$response = XF::app()->http()->client()->post($this->getApiEndpoint() . 'initiate-payment', [
				$this->getVariableName() => $payment
			]);
		}
		catch (Exception $e)
		{
			XF::logException($e);

			return $controller->error(XF::phrase('something_went_wrong_please_try_again'));
		}

		if ($response)
		{
			$responseData = json_decode($response->getBody()->getContents(), true);

			if (!empty($responseData['redirect_url']))
			{
				return $controller->redirect($responseData['redirect_url']);
			}
		}

		return $controller->error(XF::phrase('something_went_wrong_please_try_again'));
	}

	/**
	 * @param Request $request
	 *
	 * @return CallbackState
	 */
	public function setupCallback(Request $request): CallbackState
	{
		$state = new CallbackState();

		$state->providerId = $request->filter('_xfProvider', 'str');

		$state->inputRaw = $request->getInputRaw();
		$state->inputLogs = $request->getInputForLogs();

		$state->inputFiltered = $request->getInputFilterer()->filter($state->inputRaw, 'json-array');

		$state->transactionId = $request->filter('request_key', 'str');
		$state->requestKey = $request->filter('request_key', 'str');

		$state->signature = $state->inputFiltered['sign'] ?? null;
		$state->ip = $request->getIp();

		$state->httpCode = 200;

		return $state;
	}

	/**
	 * @param CallbackState $state
	 *
	 * @return bool
	 */
	public function validateCallback(CallbackState $state): bool
	{
		if ($state->transactionId && $state->requestKey)
		{
			if ($state->getPaymentProfile()->provider_id == $state->providerId)
			{
				return parent::validateCallback($state);
			}

			$state->logType = 'info';
			$state->logMessage = 'Invalid provider.';

			return false;
		}

		$state->logType = 'info';
		$state->logMessage = 'No Transaction ID or Request Key. No action to take.';

		return false;
	}

	/**
	 * @param CallbackState $state
	 *
	 * @return bool
	 */
	public function validatePurchasableData(CallbackState $state): bool
	{
		$paymentProfile = $state->getPaymentProfile();

		$options = $paymentProfile->options;
		if (!empty($options['client_secret']))
		{
			if (!empty($state->signature))
			{
				$generatedSignature = hash_hmac('sha256', implode('|', [
					$state->inputFiltered['timestamp'],
					$state->inputFiltered['subtotal'],
					$state->inputFiltered['percentage'],
					$state->inputFiltered['charge_percentage'],
					$state->inputFiltered['charge_fixed'],
					$state->inputFiltered['total']
				]), $options['client_secret']);

				if (hash_equals($generatedSignature, $state->signature))
				{
					return true;
				}

				$state->logType = 'error';
				$state->logMessage = 'Invalid signature.';

				return false;
			}

			$state->logType = 'error';
			$state->logMessage = 'Empty signature.';

			return false;
		}

		$state->logType = 'error';
		$state->logMessage = 'Invalid token.';

		return false;
	}

	/**
	 * @param CallbackState $state
	 */
	public function getPaymentResult(CallbackState $state): void
	{
		$state->paymentResult = CallbackState::PAYMENT_RECEIVED;
	}

	/**
	 * @param CallbackState $state
	 */
	public function prepareLogData(CallbackState $state): void
	{
		$state->logDetails = [
			'ip'            => $state->ip,
			'requestTime'   => XF::$time,
			'inputRaw'      => $state->inputRaw,
			'inputGetPost'  => $state->inputLogs,
			'inputFiltered' => $state->inputFiltered,
		];
	}

	/**
	 * @var array
	 */
	protected $supportedCurrencies = [
		'USD', 'EUR', 'GBP', 'RUB'
	];

	/**
	 * @param PaymentProfile $paymentProfile
	 * @param                $unit
	 * @param                $amount
	 * @param int            $result
	 *
	 * @return bool
	 */
	public function supportsRecurring(PaymentProfile $paymentProfile, $unit, $amount, &$result = self::ERR_NO_RECURRING): bool
	{
		return false;
	}

	/**
	 * @return array
	 */
	protected function getSupportedRecurrenceRanges(): array
	{
		return [];
	}

	/**
	 * @param PaymentProfile $paymentProfile
	 * @param                $currencyCode
	 *
	 * @return bool
	 */
	public function verifyCurrency(PaymentProfile $paymentProfile, $currencyCode): bool
	{
		$addOns = XF::app()->container('addon.cache');

		if (!empty($paymentProfile->options['tc_cu_exchange_to'])
			&& array_key_exists('TC/CurrencyUtils', $addOns)
			&& XF::registry()->exists('tcCurrencies'))
		{
			return array_key_exists($currencyCode, XF::registry()->get('tcCurrencies'));
		}

		return in_array($currencyCode, $this->supportedCurrencies);
	}
}
