<?php
/*
Plugin Name: CrocoPay for WooCommerce
Description: Integrate CrocoPay payment gateway with WooCommerce.
Version: 1.1
Author: Aleksei Delkhman
*/

defined('ABSPATH') || exit;

// Add CrocoPay gateway to WooCommerce
add_filter('woocommerce_payment_gateways', 'add_crocopay_gateway');

function add_crocopay_gateway($methods) {
    $methods[] = 'WC_CrocoPay_Gateway';
    return $methods;
}

// Initialize the gateway
add_action('plugins_loaded', 'init_crocopay_gateway');

function init_crocopay_gateway() {
    class WC_CrocoPay_Gateway extends WC_Payment_Gateway {
        public function __construct() {
            $this->id = 'crocopay';
            $this->has_fields = false;
            $this->method_title = 'CrocoPay';
            $this->method_description = 'Оплата через CrocoPay';

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->client_id = $this->get_option('client_id');
            $this->client_secret = $this->get_option('client_secret');
            $this->api_key = $this->get_option('api_key');
            $this->failure_page_id = $this->get_option('failure_page_id');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_wc_crocopay_gateway', array($this, 'crocopay_callback_handler'));
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => 'Включить/Выключить',
                    'type' => 'checkbox',
                    'label' => 'Включить оплату через CrocoPay',
                    'default' => 'yes',
                ),
                'title' => array(
                    'title' => 'Заголовок',
                    'type' => 'text',
                    'description' => 'Этот заголовок будет показан пользователю при оформлении заказа.',
                    'default' => 'CrocoPay',
                    'desc_tip' => true,
                ),
                'description' => array(
                    'title' => 'Описание',
                    'type' => 'textarea',
                    'description' => 'Это описание будет показано пользователю при оформлении заказа.',
                    'default' => 'Оплата через CrocoPay',
                ),
                'client_id' => array(
                    'title' => 'Client ID',
                    'type' => 'text',
                    'description' => 'Введите ваш Client ID CrocoPay.',
                    'default' => '',
                ),
                'client_secret' => array(
                    'title' => 'Client Secret',
                    'type' => 'text',
                    'description' => 'Введите ваш Client Secret CrocoPay.',
                    'default' => '',
                ),
                'failure_page_id' => array(
                    'title' => 'Страница неуспешного платежа',
                    'type' => 'select',
                    'description' => 'Выберите страницу, на которую будет перенаправлен пользователь в случае неуспешного платежа.',
                    'options' => $this->get_pages(),
                    'default' => '',
                ),
            );
        }

        private function get_pages() {
            $pages = get_pages();
            $options = array();
            foreach ($pages as $page) {
                $options[$page->ID] = $page->post_title;
            }
            return $options;
        }

        public function process_payment($order_id) {
            $order = wc_get_order($order_id);
            $successp = $order->get_checkout_order_received_url();
            $callb = get_permalink($this->failure_page_id);

            $data = array(
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'amount' => (string)$order->get_total(),
                'currency' => get_woocommerce_currency(),
                'successUrl' => $successp,
                'cancelUrl' => $callb,
                'callbackUrl' => home_url('/wc-api/WC_CrocoPay_Gateway'),
            );

            $response = wp_remote_post('https://crocopay.tech/api/v2/initiate-payment', array(
                'method' => 'POST',
                'body' => $data,
                'headers' => array(
                   // 'Authorization' => 'Bearer ' . $this->api_key,
                ),
            ));

            if (is_wp_error($response)) {
                wc_add_notice(__('Ошибка платежа:', 'crocopay-woocommerce') . ' ' . $response->get_error_message(), 'error');
                return;
            }

            $response_body = wp_remote_retrieve_body($response);
            $response_data = json_decode($response_body, true);

            if ($response_data && isset($response_data['status']) && $response_data['status'] === 'success') {
                $order->update_status('on-hold', __('Ожидание оплаты через CrocoPay', 'crocopay-woocommerce'));
                wc_reduce_stock_levels($order_id);
                WC()->cart->empty_cart();

                return array(
                    'result' => 'success',
                    'redirect' => $response_data['redirect_url'],
                );
            } else {
                wc_add_notice(__('Ошибка платежа:', 'crocopay-woocommerce') . ' ' . $response_data['message'], 'error');
                return;
            }
        }

    }
}
